/*
 * Copyright (C) 2008-2013 J.Rios <anonbeat@gmail.com>
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth 
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 * 	Jörn Magens <shuerhaaken@googlemail.com>
 * 	Matias De lellis <mati86dl@gmail.com>
 * 	Pavel Vasin <rat4vier@gmail.com>
 */




#include "taginfo.h"
#include "taginfo_internal.h"
#include "tstring.h"
#include "tag.h"
#include "taglib.h"
#include "tbytevector.h"

#define COMPILATION_FLAG         "WM/PartOfACompilation"


using namespace TagInfo;



bool get_asf_image(ASF::Tag * asftag, char*& data, int &data_length, ImageType &image_type) {
    data = NULL;
    data_length = 0;
    image_type = IMAGE_TYPE_UNKNOWN;
    if(asftag) {
        if(asftag->attributeListMap().contains("WM/Picture")) {
            ASF::Attribute attr = asftag->attributeListMap()["WM/Picture"].front();
            ASF::Picture pict = attr.toPicture();
            if(!pict.isValid())
                return false;
            
            ByteVector PictureData = pict.picture();
            
            if(PictureData.size() > 0) {
                data_length = PictureData.size();
                data = new char[data_length];
                memcpy(data, PictureData.data(), PictureData.size());
                
                String mimetype = pict.mimeType();
                if(mimetype.find("/jpeg") != -1 || mimetype.find("/jpg") != -1)
                    image_type = IMAGE_TYPE_JPEG;
                else if(mimetype.find("/png") != -1)
                    image_type = IMAGE_TYPE_PNG;
                
                return true;
            }
        }
    }
    return false;
}

bool set_asf_image(ASF::Tag * asftag, const char* data, int data_length, ImageType image_type) {
    if(asftag) {
        if(asftag->attributeListMap().contains("WM/Picture"))
            asftag->removeItem("WM/Picture");
        
        if(!data || data_length == 0)
            return true;
        
        ASF::Picture picture = ASF::Picture();
        if(image_type == IMAGE_TYPE_JPEG || image_type == IMAGE_TYPE_UNKNOWN) //default to jpeg
            picture.setMimeType("image/jpeg");
        if(image_type == IMAGE_TYPE_PNG)
            picture.setMimeType("image/png");
        
        ByteVector ImgData = ByteVector((TagLib::uint) data_length);
        memcpy(ImgData.data(), data, data_length);
        
        picture.setPicture(ImgData);
        ASF::Attribute attr = ASF::Attribute(picture);
        asftag->setAttribute("WM/Picture", attr);
        
        return true;
    }
    return false;
}

ASFInfo::ASFInfo(const String &filename) : Info(filename) {
    if(!file_name.isEmpty() && !create_file_ref())
        printf("Error creating file ref! %s\n", filename.toCString());
    if(taglib_file) {
        asf_tag = ((TagLib::ASF::File *) taglib_file)->tag();
        if(!asf_tag)
            valid = false;
    }
    else {
        printf("Cant get xiphcomment from '%s'\n", file_name.toCString(false));
        asf_tag = NULL;
        valid = false;
    }
}


ASFInfo::~ASFInfo() {

}


bool ASFInfo::create_file_ref() {
    if(file_name.isEmpty())
        return false;
    taglib_file = new TagLib::ASF::File(file_name.toCString(false), true, TagLib::AudioProperties::Fast);
    if(taglib_file) {
        return true;
    }
    else {
        printf("TagLib::File creation failed for '%s'\n", file_name.toCString(false));
        return false;
    }
//    if(!file_name.isEmpty())
//        taglib_fileref = new TagLib::FileRef(file_name.toCString(false), true, TagLib::AudioProperties::Fast);
//    else
//        return false;
//    
//    if(taglib_fileref && !taglib_fileref->isNull()) {
//        taglib_tag = taglib_fileref->tag();
//        if(!taglib_tag) {
//            printf("Cant get tag object from '%s'\n", file_name.toCString(false));
//            return false;
//        }
//        return true;
//    }
//    else {
//        return false;
//    }
}


bool ASFInfo::read(void) {
    if(Info::read()) {
        if(asf_tag) {
            read_virtual_tags((TagLib::Tag *)asf_tag);
            
            if(asf_tag->attributeListMap().contains("WM/PartOfSet")) {
                disk_number = 
                    atol(asf_tag->attributeListMap()[ "WM/PartOfSet" ].front().toString().toCString(false));
                if(disk_number <= 0)
                    disk_number = 1;
            }
            if(asf_tag->attributeListMap().contains("WM/Composer")) {
                composer = asf_tag->attributeListMap()[ "WM/Composer" ].front().toString();
            }
            if(asf_tag->attributeListMap().contains(COMPILATION_FLAG)) {
                is_compilation =
                    asf_tag->attributeListMap()[COMPILATION_FLAG].front().toString() == String("1");
            }
            if(asf_tag->attributeListMap().contains("WM/AlbumArtist")) {
                album_artist = asf_tag->attributeListMap()[ "WM/AlbumArtist" ].front().toString();
            }
            long rat = 0;
#if 0 // valgrind
            if(asf_tag->attributeListMap().contains("WM/SharedUserRating")) {
                rat = atol(asf_tag->attributeListMap()[ "WM/SharedUserRating" ].front().toString().toCString(false));
            }
            if(!rat && asf_tag->attributeListMap().contains("Rating")) {
                rat = atol(asf_tag->attributeListMap()[ "Rating" ].front().toString().toCString(false));
            }
#endif // valgrind
            if(rat) {
                if(rat > 5)
                {
                    rating = wm_rating_to_rating(rat);
                }
                else
                {
                    rating = rat;
                }
            }
            if(track_labels.size() == 0) {
                if(asf_tag->attributeListMap().contains("TRACK_LABELS")){
                    track_labels_string = asf_tag->attributeListMap()[ "TRACK_LABELS" ].front().toString();
                }
            }
            if(artist_labels.size() == 0) {
                if(asf_tag->attributeListMap().contains("ARTIST_LABELS"))
                {
                    artist_labels_string = asf_tag->attributeListMap()[ "ARTIST_LABELS" ].front().toString();
//                    artist_labels = artist_labels_string.split("|");
                }
            }
            if(album_labels.size() == 0) {
                if(asf_tag->attributeListMap().contains("ALBUM_LABELS"))
                {
                    album_labels_string = asf_tag->attributeListMap()[ "ALBUM_LABELS" ].front().toString();
//                    album_labels = album_labels_string.split("|");
                }
            }
        }
    }
    else {
        printf("Error: Could not read tags from file '%s'\n", file_name.toCString(true));
        return false; 
    }
    return true;
}


void check_asf_label_frame(ASF::Tag * asftag, const char * description, const String &value) {
    if(asftag->attributeListMap().contains(description))
        asftag->removeItem(description);
    if(!value.isEmpty()) {
            asftag->setAttribute(description, value);
    }
}


bool ASFInfo::write(void) {
    if(asf_tag) {
        if(changedflag) {
            if(changedflag & CHANGED_DATA_DISK_NUM) {
                asf_tag->removeItem("WM/PartOfSet");
                asf_tag->setAttribute("WM/PartOfSet", format("%u", disk_number));
            }
            if(changedflag & CHANGED_COMPOSER_TAG) {
                asf_tag->removeItem("WM/Composer");
                asf_tag->setAttribute("WM/Composer", composer);
            }
            if(changedflag & CHANGED_IS_COMPILATION_TAG) {
                asf_tag->removeItem(COMPILATION_FLAG);
                if(is_compilation)
                    asf_tag->setAttribute(COMPILATION_FLAG, String("1"));
                else
                    asf_tag->setAttribute(COMPILATION_FLAG, String("0"));
            }
            if(changedflag & CHANGED_DATA_ALBUMARTIST) {
                asf_tag->removeItem("WM/AlbumArtist");
                asf_tag->setAttribute("WM/AlbumArtist", album_artist);
            }
            if(changedflag & CHANGED_DATA_RATING) {
                asf_tag->removeItem("WM/SharedUserRating");
                int WMRatings[] = { 0, 0, 1, 25, 50, 75, 99 };
                asf_tag->setAttribute("WM/SharedUserRating", format("%i", WMRatings[ rating + 1 ]));
            }
            // The Labels
            if(changedflag & CHANGED_TRACK_LABELS)
                check_asf_label_frame(asf_tag, "TRACK_LABELS", track_labels_string);
            if(changedflag & CHANGED_ARTIST_LABELS)
                check_asf_label_frame(asf_tag, "ARTIST_LABELS", artist_labels_string);
            if(changedflag & CHANGED_ALBUM_LABELS)
                check_asf_label_frame(asf_tag, "ALBUM_LABELS", album_labels_string);
            write_virtual_tags((TagLib::Tag *)asf_tag);
        }
    }
    return Info::write();
}


bool ASFInfo::can_handle_images(void) const{
    return true;
}

bool ASFInfo::get_image(char*& data, int &data_length, ImageType &image_type) const {
    if(asf_tag) {
        return get_asf_image(asf_tag, data, data_length, image_type);
    }
    return false;
}

bool ASFInfo::set_image(const char* data, int data_length, ImageType image_type) {
    if(asf_tag) {
        return set_asf_image(asf_tag, data, data_length, image_type);
    }
    return false;
}


bool ASFInfo::can_handle_lyrics(void) const {
    return true;
}


String ASFInfo::get_lyrics(void) const {
    if(asf_tag) {
            if(asf_tag->attributeListMap().contains("WM/Lyrics")) {
            return asf_tag->attributeListMap()[ "WM/Lyrics" ].front().toString();
        }
    }
    return String();
}


bool ASFInfo::set_lyrics(const String &lyrics) {
    if(asf_tag) {
        asf_tag->removeItem("WM/Lyrics");
        asf_tag->setAttribute("WM/Lyrics", lyrics);
        return true;
    }
    return false;
}

bool ASFInfo::get_encrypted(void) const
{
    if(taglib_file) {
        TagLib::ASF::Properties *properties = (TagLib::ASF::Properties*)taglib_file->audioProperties();

        if(properties) {
            return properties->isEncrypted();
        }
    }

    return false;
}


